/*
* FileSaver.js
* A saveAs() FileSaver implementation.
*
* By Eli Grey, http://eligrey.com
*
* License : https://github.com/eligrey/FileSaver.js/blob/master/LICENSE.md (MIT)
* source  : http://purl.eligrey.com/github/FileSaver.js
*/

// The one and only way of getting global scope in all environments
// https://stackoverflow.com/q/3277182/1008999
var _global = typeof window === 'object' && window.window === window
  ? window : typeof self === 'object' && self.self === self
  ? self : typeof global === 'object' && global.global === global
  ? global
  : this

function bom (blob, opts) {
  if (typeof opts === 'undefined') opts = { autoBom: false }
  else if (typeof opts !== 'object') {
    console.warn('Deprecated: Expected third argument to be a object')
    opts = { autoBom: !opts }
  }

  // prepend BOM for UTF-8 XML and text/* types (including HTML)
  // note: your browser will automatically convert UTF-16 U+FEFF to EF BB BF
  if (opts.autoBom && /^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(blob.type)) {
    return new Blob([String.fromCharCode(0xFEFF), blob], { type: blob.type })
  }
  return blob
}

function download (url, name, opts) {
  var xhr = new XMLHttpRequest()
  xhr.open('GET', url)
  xhr.responseType = 'blob'
  xhr.onload = function () {
    saveAs(xhr.response, name, opts)
  }
  xhr.onerror = function () {
    console.error('could not download file')
  }
  xhr.send()
}

function corsEnabled (url) {
  var xhr = new XMLHttpRequest()
  // use sync to avoid popup blocker
  xhr.open('HEAD', url, false)
  try {
    xhr.send()
  } catch (e) {}
  return xhr.status >= 200 && xhr.status <= 299
}

// `a.click()` doesn't work for all browsers (#465)
function click (node) {
  try {
    node.dispatchEvent(new MouseEvent('click'))
  } catch (e) {
    var evt = document.createEvent('MouseEvents')
    evt.initMouseEvent('click', true, true, window, 0, 0, 0, 80,
                          20, false, false, false, false, 0, null)
    node.dispatchEvent(evt)
  }
}

// Detect WebView inside a native macOS app by ruling out all browsers
// We just need to check for 'Safari' because all other browsers (besides Firefox) include that too
// https://www.whatismybrowser.com/guides/the-latest-user-agent/macos
var isMacOSWebView = _global.navigator && /Macintosh/.test(navigator.userAgent) && /AppleWebKit/.test(navigator.userAgent) && !/Safari/.test(navigator.userAgent)

var saveAs = _global.saveAs || (
  // probably in some web worker
  (typeof window !== 'object' || window !== _global)
    ? function saveAs () { /* noop */ }

  // Use download attribute first if possible (#193 Lumia mobile) unless this is a macOS WebView
  : ('download' in HTMLAnchorElement.prototype && !isMacOSWebView)
  ? function saveAs (blob, name, opts) {
    var URL = _global.URL || _global.webkitURL
    // Namespace is used to prevent conflict w/ Chrome Poper Blocker extension (Issue #561)
    var a = document.createElementNS('http://www.w3.org/1999/xhtml', 'a')
    name = name || blob.name || 'download'

    a.download = name
    a.rel = 'noopener' // tabnabbing

    // TODO: detect chrome extensions & packaged apps
    // a.target = '_blank'

    if (typeof blob === 'string') {
      // Support regular links
      a.href = blob
      if (a.origin !== location.origin) {
        corsEnabled(a.href)
          ? download(blob, name, opts)
          : click(a, a.target = '_blank')
      } else {
        click(a)
      }
    } else {
      // Support blobs
      a.href = URL.createObjectURL(blob)
      setTimeout(function () { URL.revokeObjectURL(a.href) }, 4E4) // 40s
      setTimeout(function () { click(a) }, 0)
    }
  }

  // Use msSaveOrOpenBlob as a second approach
  : 'msSaveOrOpenBlob' in navigator
  ? function saveAs (blob, name, opts) {
    name = name || blob.name || 'download'

    if (typeof blob === 'string') {
      if (corsEnabled(blob)) {
        download(blob, name, opts)
      } else {
        var a = document.createElement('a')
        a.href = blob
        a.target = '_blank'
        setTimeout(function () { click(a) })
      }
    } else {
      navigator.msSaveOrOpenBlob(bom(blob, opts), name)
    }
  }

  // Fallback to using FileReader and a popup
  : function saveAs (blob, name, opts, popup) {
    // Open a popup immediately do go around popup blocker
    // Mostly only available on user interaction and the fileReader is async so...
    popup = popup || open('', '_blank')
    if (popup) {
      popup.document.title =
      popup.document.body.innerText = 'downloading...'
    }

    if (typeof blob === 'string') return download(blob, name, opts)

    var force = blob.type === 'application/octet-stream'
    var isSafari = /constructor/i.test(_global.HTMLElement) || _global.safari
    var isChromeIOS = /CriOS\/[\d]+/.test(navigator.userAgent)

    if ((isChromeIOS || (force && isSafari) || isMacOSWebView) && typeof FileReader !== 'undefined') {
      // Safari doesn't allow downloading of blob URLs
      var reader = new FileReader()
      reader.onloadend = function () {
        var url = reader.result
        url = isChromeIOS ? url : url.replace(/^data:[^;]*;/, 'data:attachment/file;')
        if (popup) popup.location.href = url
        else location = url
        popup = null // reverse-tabnabbing #460
      }
      reader.readAsDataURL(blob)
    } else {
      var URL = _global.URL || _global.webkitURL
      var url = URL.createObjectURL(blob)
      if (popup) popup.location = url
      else location.href = url
      popup = null // reverse-tabnabbing #460
      setTimeout(function () { URL.revokeObjectURL(url) }, 4E4) // 40s
    }
  }
)

_global.saveAs = saveAs.saveAs = saveAs

if (typeof module !== 'undefined') {
  module.exports = saveAs;
}



/*
---------------------
Rallye Sciences 2024
---------------------
*/

var DEVMODE = false; //set this to FALSE to remove developper options (reset data storage)

var data = {};

/* evaluator are structured as follows : 
{
    [activityid] : {
        [fieldId] : {coef : number, evaluator : function}
        
    }
}

The evaluator is a function that take in parameter the field value as a string,
and returns :
- "i" if the value is invalid (e.g expected an int got something else)
- "b" if the field is graded bronze.
- "s" if the field is graded silver.
- "g" if the field is graded gold.
*/

function isnum(s) {
    //code from https://stackoverflow.com/a/1779019
    return /^\d+$/.test(val);
}

function svalue(fv) {
    return fv
}

function expected(v) {
    return (fv) => {
        i = parseInt(fv);
        if (isNaN(i)) {
            return "i"
        }
        if (i === v) {
            return "g"
        } else {
            return "b"
        }
    }
}

var qr_matrix_coef = 3;

function eval_qr_matrix() {
    qr_err = check_qr_matrix()

    if (qr_err === 0) {
        return "g";
    } else if (qr_err < 20) {
        return "s";
    } else {
        return "b";
    }
}


var evaluators = {
    "activity-1": {
        "name2": {
            coef: 1,
            evaluator: (fv) => { return (fv === "valeur") ? "g" : "i"; }
        },
        "select2": {
            coef: 2,
            evaluator: (fv) => { return fv; }
        }
    },
    "activity-2": {
        "name3": {
            coef: 1,
            evaluator: (fv) => { return (fv === "chiffre") ? "g" : "i"; }
        },
        "select3": {
            coef: 2,
            evaluator: (fv) => { return fv; }
        }
    },
    "activity-3": {
        "bpm": {
            coef: 1,
            evaluator: (fv) => {
                i = parseInt(fv);
                if (isNaN(i)) {
                    return "i"
                }
                if (i === 120) {
                    return "g"
                } if (40 < i && i < 180) {
                    return "s"
                }
                return "b"
            }
        },

        "mesure-1": {
            coef: 1,
            evaluator: svalue
        },
        "mesure-2": {
            coef: 1,
            evaluator: svalue
        },
        "mesure-3": {
            coef: 1,
            evaluator: svalue
        },
        "mesure-4": {
            coef: 1,
            evaluator: svalue
        },
        "mesure-5": {
            coef: 1,
            evaluator: svalue
        },
        "mesure-6": {
            coef: 1,
            evaluator: svalue
        },
        "mesure-7": {
            coef: 1,
            evaluator: svalue
        },

        "periode-1": {
            coef: 1,
            evaluator: svalue
        },
        "periode-2": {
            coef: 1,
            evaluator: svalue
        },
        "periode-3": {
            coef: 1,
            evaluator: svalue
        },

        "moyenne-a": {
            coef: 1,
            evaluator: expected(8)
        },
        "moyenne-b": {
            coef: 1,
            evaluator: expected(6)
        },
        "moyenne-c": {
            coef: 1,
            evaluator: expected(4)
        },
    }
};

function evaluate_field(activityId, fieldId) {
    eval = "i";
    if ((data[activityId] && data[activityId][fieldId])) {
        if (!(evaluators[activityId] && evaluators[activityId][fieldId])) {
            console.error(`Field ${fieldId} in activity ${activityId} has no evaluator defined !`);
            eval = "i";
        } else {
            eval = evaluators[activityId][fieldId].evaluator(data[activityId][fieldId]);
            if (!(["i", "b", "s", "g"].includes(eval))) {
                console.error(`Evaluator for field ${fieldId} in activity ${activityId} returned an invalid value : ${eval} !`);
            }
        }
    }

    const field = document.getElementById(fieldId);

    if (eval === "i") {
        field.classList.add("invalid");
    } else {
        field.classList.remove("invalid");
    }


    return [eval, ((evaluators[activityId] && evaluators[activityId][fieldId]) ? evaluators[activityId][fieldId].coef : 1)];

}

const EVALUATED_ID = "__evaluated__";

function majority(b, s, g) {
    if (g >= s && g >= b) {
        return "g";
    } else if (s > g && s >= b) {
        return "s";
    } else {
        return "b";
    }
}

function evaluate_activity(activity, force = false) {


    id = activity.getAttribute("id");

    if (data[id] && data[id][EVALUATED_ID] && !force) {
        return;
    }

    let score = {
        i: 0, b: 0, s: 0, g: 0
    }

    activity.querySelectorAll("input").forEach((input) => {
        const inputId = input.getAttribute("id");
        const [eval, coef] = evaluate_field(id, inputId);
        score[eval] += coef;
    });

    activity.querySelectorAll("select").forEach((select) => {
        const selectId = select.getAttribute("id");
        const [eval, coef] = evaluate_field(id, selectId);
        score[eval] += coef;
    });


    if (id === "activity-2") {
        eval = eval_qr_matrix();
        score[eval] += qr_matrix_coef;
    }


    console.log("SCORE : ");
    console.log(score);

    if (score.i === 0) {
        const classes = { "g": "gold", "s": "silver", "b": "bronze" }
        const texts = { "g": "Or", "s": "Argent", "b": "Bronze" }

        const res = classes[majority(score.b, score.s, score.g)]
        register_change(id, EVALUATED_ID, res);

        activity.querySelector(".validation").innerText = texts[majority(score.b, score.s, score.g)];
        activity.classList.add(res);

        activity.querySelectorAll("input").forEach((input) => {
            input.disabled = true;
        });

        activity.querySelectorAll("select").forEach((select) => {
            select.disabled = true;
        });
    }
}

const DATA_STORAGE_NAME = "rls";

function register_change(activityId, id, answer) {
    const d = JSON.parse(localStorage.getItem(DATA_STORAGE_NAME));

    if (!d[activityId]) {
        d[activityId] = {};
    }
    d[activityId][id] = answer;

    d;

    localStorage.setItem(DATA_STORAGE_NAME, JSON.stringify(d));
}

function init_activity(activity) {

    const id = activity.getAttribute("id");
    valid = false;
    if (data[id] && data[id][EVALUATED_ID]) {
        evaluate_activity(activity, true);
    }

    activity.querySelectorAll("input").forEach(input => {
        init_input(activity, input);
    });
    activity.querySelectorAll("textarea").forEach(textarea => {
        init_input(activity, textarea);
    });

    activity.querySelectorAll("select").forEach(select => {
        init_select(activity, select);
    })
}

function init_input(activity, input) {

    let value = "";
    let activityId = activity.getAttribute("id");
    let id = input.getAttribute("id");

    console.log("trying to set");
    console.log(activityId);
    console.log(id);

    if (data[activityId] && data[activityId][id]) {
        console.log("FOUND IT");
        value = data[activityId][id]
    }

    input.value = value;

    input.addEventListener("input",
        function (evt) {
            register_change(activityId, id, this.value);
        });

    input.addEventListener("textarea",
        function (evt) {
            register_change(activityId, id, this.value);
        });
}

function init_select(activity, select) {
    value = "";
    let activityId = activity.getAttribute("id");
    let id = select.getAttribute("id");

    if (data[activityId] && data[activityId][id]) {
        console.log("FOUND IT");
        value = data[activityId][id]
    }

    select.value = value;

    select.addEventListener("change",
        function (evt) {
            register_change(activityId, id, select.value);
        });
}

function init() {

    if (DEVMODE) {
        let reset = document.getElementById("title");
        reset.addEventListener("click", () => {
            console.log("cleared storage !");
            localStorage.clear();
        });
    }

    //recovering storage if it exists
    const d = localStorage.getItem(DATA_STORAGE_NAME);

    if (d != null) {
        console.log("DATA FROM STORAGE");
        console.log(d)
        data = JSON.parse(d);
    } else {
        localStorage.setItem(DATA_STORAGE_NAME, JSON.stringify({}));
    }

    console.log("DATA");
    console.log(data);



    document.querySelectorAll(".activity").forEach((activity) => {
        init_activity(activity)
    })

    //initializing each questionnaire
}

function get_export_field(id) {

    input = document.getElementById(id)
    const val = input.value;

    regex = /^[a-zA-Z0-9](?:\ ?[a-zA-Z0-9])*$/gs

    res = val.match(regex);

    if (res === null || res.length != 1) {
        input.classList.add("invalid");
        return null
    }
    input.classList.remove("invalid");
    return val;
}

function export_results() {

    classe = get_export_field("classe");
    etablissement = get_export_field("etablissement");

    if (classe === null || etablissement === null) {
        return;
    }


    const data = localStorage.getItem(DATA_STORAGE_NAME);

    const res = {

        "etablissement": etablissement,
        "classe": classe,
        "results": JSON.parse(data),
    }

    filename = "";
    filename += etablissement.toLowerCase().replaceAll(" ", "_");
    filename += "__"
    filename += classe.toLowerCase().replaceAll(" ", "_");;
    filename += "__"
    filename += "reponses.json";

    var blob = new Blob([JSON.stringify(res)], { type: "text/plain;charset=utf-8" });
    saveAs(blob, filename);
}